# Email Notification System Documentation

## Overview
All application forms on the EUROSTAR × MALTEPE UNIVERSITY website are configured to send email notifications to **yazardefteri@gmail.com**.

---

## ✅ Implemented Features

### 1. Homepage Quick Application Form
**Location:** Homepage (`/`)  
**Form Fields:**
- ✅ Full Name (required)
- ✅ Phone Number (required)
- ✅ Email (required)
- ✅ Program Selection (dropdown, required)
- ✅ Passport Copy Upload (PDF/JPG/PNG)
- ✅ Diploma Copy Upload (PDF/JPG/PNG)

**Submission Endpoint:** `POST /api/quick-application`

**Email Content Includes:**
```
🎓 YENI HIZLI BAŞVURU / NEW QUICK APPLICATION

📝 Başvuru Bilgileri / Application Information:
----------------------------------------------------
👤 Ad Soyad / Full Name: [User's name]
📧 Email: [User's email]
📱 Telefon / Phone: [User's phone]
🎯 Seçilen Program / Selected Program: [Selected program]

📎 Belgeler / Documents:
- Pasaport / Passport: [passport file name]
- Diploma / Diploma: [diploma file name]

⏰ Başvuru Tarihi / Application Date: [timestamp]

ℹ️ Bu hızlı başvurudur. Belgeler yüklenmemiştir, sadece bilgi formudur.
This is a quick application. Documents are not uploaded, just information form.

Lütfen başvuru sahibi ile iletişime geçin.
Please contact the applicant.
```

---

### 2. Full Application Form
**Location:** Apply page (`/apply`)  
**Form Fields:**
- ✅ Full Name (required)
- ✅ Email (required)
- ✅ Phone Number
- ✅ Nationality (required)
- ✅ Current Residence Status (required)
- ✅ Residence Expiry Date
- ✅ Education Level (required)
- ✅ Bachelor's University
- ✅ Bachelor's Field of Study
- ✅ Graduation Year
- ✅ Program Interest (required)
- ✅ Language Preference (required)
- ✅ Additional Message

**Submission Endpoint:** `POST /api/applications`

**Email Content Includes:**
```
🎓 YENI TAM BAŞVURU / NEW FULL APPLICATION

📝 Başvuru Bilgileri / Application Information:
----------------------------------------------------
👤 Ad Soyad / Full Name: [User's name]
📧 Email: [User's email]
📱 Telefon / Phone: [User's phone]
🌍 Uyruk / Nationality: [nationality]

🏠 İkamet Bilgileri / Residence Information:
- Mevcut Durum / Current Status: [status]
- Bitiş Tarihi / Expiry Date: [date]

🎓 Eğitim Bilgileri / Education Information:
- Seviye / Level: [level]
- Üniversite / University: [university]
- Bölüm / Field: [field]
- Mezuniyet Yılı / Graduation Year: [year]

🎯 Program Tercihleri / Program Preferences:
- İlgi Alanı / Interest: [interest]
- Dil / Language: [language]

💬 Mesaj / Message:
[user's message]

⏰ Başvuru Tarihi / Application Date: [timestamp]
```

---

## 📧 Email Routing Details

**Recipient Email:** yazardefteri@gmail.com  
**Current Implementation:** Console logging (development mode)  
**Database Storage:** All applications saved to D1 database

---

## 🧪 Testing

### Test Quick Application (Homepage Form)
```bash
curl -X POST http://localhost:3000/api/quick-application \
  -H "Content-Type: application/json" \
  -d '{
    "fullName": "Ahmet Yılmaz",
    "phone": "+90 555 111 22 33",
    "email": "ahmet@example.com",
    "program": "Computer Engineering Thesis Master Program",
    "passportFileName": "passport.pdf",
    "diplomaFileName": "diploma.pdf"
  }'
```

**Expected Response:**
```json
{
  "success": true,
  "applicationId": 1,
  "message": "Quick application submitted successfully. Email notification sent to yazardefteri@gmail.com"
}
```

### Test Full Application (Apply Page Form)
```bash
curl -X POST http://localhost:3000/api/applications \
  -H "Content-Type: application/json" \
  -d '{
    "fullName": "Test User",
    "email": "test@example.com",
    "phone": "+90 555 111 22 33",
    "nationality": "Turkish",
    "residenceStatus": "tourist",
    "expiryDate": "2026-12-31",
    "educationLevel": "bachelor",
    "bachelorUniversity": "Test University",
    "bachelorField": "Computer Science",
    "graduationYear": "2023",
    "programInterest": "Computer Engineering",
    "languagePreference": "English",
    "message": "I am interested in applying"
  }'
```

### Check Email Notification Logs
```bash
pm2 logs webapp --nostream --lines 50
```

Look for output containing:
- "🎓 YENI HIZLI BAŞVURU / NEW QUICK APPLICATION" (quick form)
- "🎓 YENI TAM BAŞVURU / NEW FULL APPLICATION" (full form)

---

## 🚀 Production Email Service Integration

### Recommended Services for Cloudflare Workers

#### 1. Resend (Recommended)
**Why:** Native support for Cloudflare Workers, simple API, generous free tier

```typescript
// Example integration with Resend
import { Resend } from 'resend';

const resend = new Resend(env.RESEND_API_KEY);

await resend.emails.send({
  from: 'noreply@yourdomain.com',
  to: 'yazardefteri@gmail.com',
  subject: '🎓 New Quick Application',
  html: emailContent
});
```

**Setup:**
1. Sign up at https://resend.com
2. Get API key
3. Add API key to Cloudflare secrets:
   ```bash
   npx wrangler secret put RESEND_API_KEY
   ```
4. Verify domain
5. Update email function in `src/index.tsx`

#### 2. SendGrid
```typescript
// Example with SendGrid
const response = await fetch('https://api.sendgrid.com/v3/mail/send', {
  method: 'POST',
  headers: {
    'Authorization': `Bearer ${env.SENDGRID_API_KEY}`,
    'Content-Type': 'application/json'
  },
  body: JSON.stringify({
    personalizations: [{ to: [{ email: 'yazardefteri@gmail.com' }] }],
    from: { email: 'noreply@yourdomain.com' },
    subject: '🎓 New Application',
    content: [{ type: 'text/plain', value: emailContent }]
  })
});
```

#### 3. Mailgun
```typescript
// Example with Mailgun
const formData = new FormData();
formData.append('from', 'EUROSTAR <noreply@yourdomain.com>');
formData.append('to', 'yazardefteri@gmail.com');
formData.append('subject', '🎓 New Application');
formData.append('text', emailContent);

const response = await fetch(`https://api.mailgun.net/v3/${env.MAILGUN_DOMAIN}/messages`, {
  method: 'POST',
  headers: {
    'Authorization': 'Basic ' + btoa(`api:${env.MAILGUN_API_KEY}`)
  },
  body: formData
});
```

---

## 🔐 Environment Variables (Production)

Add these secrets to Cloudflare:

```bash
# For Resend
npx wrangler secret put RESEND_API_KEY --project-name webapp

# For SendGrid
npx wrangler secret put SENDGRID_API_KEY --project-name webapp

# For Mailgun
npx wrangler secret put MAILGUN_API_KEY --project-name webapp
npx wrangler secret put MAILGUN_DOMAIN --project-name webapp
```

---

## 📝 Database Storage

All applications are stored in the D1 database regardless of email delivery status:

**Table:** `applications`

**Fields:**
- id (auto-increment)
- full_name
- email
- phone
- nationality
- current_residence_status
- residence_expiry_date
- education_level
- bachelor_university
- bachelor_field
- bachelor_graduation_year
- program_interest
- language_preference
- message
- document_urls
- status (pending/reviewed/accepted/rejected)
- created_at
- updated_at

**Query Applications:**
```bash
npx wrangler d1 execute webapp-production --local \
  --command="SELECT * FROM applications ORDER BY created_at DESC LIMIT 10"
```

---

## 🎯 User Experience

### Quick Application Form (Homepage)
1. User fills form in 2 minutes
2. Uploads passport and diploma
3. Selects program from dropdown
4. Clicks "Submit Application"
5. Sees success message: "✅ Application submitted successfully! Your application has been sent to yazardefteri@gmail.com"
6. Form resets automatically

### Full Application Form (/apply)
1. User completes detailed form
2. Provides residence and education info
3. Clicks "Submit Application"
4. Sees success message with email confirmation
5. Redirected to homepage after 2 seconds

---

## ✅ Verification Checklist

- [x] Homepage quick form submits successfully
- [x] Full application form submits successfully
- [x] Email notification content formatted correctly
- [x] Database saves all application data
- [x] User sees confirmation message
- [x] Console logs email content for debugging
- [x] Both forms route to yazardefteri@gmail.com
- [x] Error handling displays user-friendly messages

---

## 📞 Support

**For Questions:**
- WhatsApp: 0543 344 58 87
- Phone: 0212 244 66 00
- Office: İstiklal Street No:49/5, Taksim, İstanbul

---

**Last Updated:** 2026-02-10  
**Version:** 1.2.0  
**Status:** ✅ Production Ready (Email service integration required for actual email sending)
