// Application form JavaScript

// Get language from URL
const urlParams = new URLSearchParams(window.location.search);
const lang = urlParams.get('lang') || 'en';

// Translation keys
const translations = {
  en: {
    submitting: 'Submitting...',
    success: 'Application submitted successfully! We will contact you soon.',
    error: 'Error submitting application. Please try again.',
    required: 'Please fill in all required fields.',
    emailInvalid: 'Please enter a valid email address.'
  }
};

const tr = translations[lang] || translations.en;

// Handle form submission
document.getElementById('applicationForm')?.addEventListener('submit', async (e) => {
  e.preventDefault();
  
  const form = e.target;
  const submitButton = form.querySelector('button[type="submit"]');
  const originalText = submitButton.textContent;
  
  // Disable submit button
  submitButton.disabled = true;
  submitButton.textContent = tr.submitting;
  
  // Get form data
  const formData = {
    fullName: form.fullName.value,
    email: form.email.value,
    phone: form.phone.value,
    nationality: form.nationality.value,
    residenceStatus: form.residenceStatus.value,
    expiryDate: form.expiryDate.value,
    educationLevel: form.educationLevel.value,
    bachelorUniversity: form.bachelorUniversity.value,
    bachelorField: form.bachelorField.value,
    graduationYear: form.graduationYear.value,
    programInterest: form.programInterest.value,
    languagePreference: form.languagePreference.value,
    message: form.message.value
  };
  
  // Basic validation
  if (!formData.fullName || !formData.email || !formData.nationality || 
      !formData.residenceStatus || !formData.educationLevel || 
      !formData.programInterest || !formData.languagePreference) {
    alert(tr.required);
    submitButton.disabled = false;
    submitButton.textContent = originalText;
    return;
  }
  
  // Email validation
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(formData.email)) {
    alert(tr.emailInvalid);
    submitButton.disabled = false;
    submitButton.textContent = originalText;
    return;
  }
  
  try {
    const response = await fetch('/api/applications', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(formData)
    });
    
    const data = await response.json();
    
    if (response.ok && data.success) {
      // Show success message with email info
      alert(`✅ ${tr.success}\n\n📧 Your application has been sent to:\nyazardefteri@gmail.com\n\nWe will contact you soon!`);
      
      // Reset form
      form.reset();
      
      // Redirect to homepage after 2 seconds
      setTimeout(() => {
        window.location.href = `/?lang=${lang}`;
      }, 2000);
    } else {
      throw new Error(data.error || 'Unknown error');
    }
  } catch (error) {
    console.error('Error submitting application:', error);
    alert(tr.error);
  } finally {
    submitButton.disabled = false;
    submitButton.textContent = originalText;
  }
});

// Auto-format phone number (optional enhancement)
document.getElementById('phone')?.addEventListener('input', (e) => {
  let value = e.target.value.replace(/\D/g, '');
  if (value.length > 0) {
    if (value.length <= 3) {
      e.target.value = value;
    } else if (value.length <= 6) {
      e.target.value = value.slice(0, 3) + ' ' + value.slice(3);
    } else {
      e.target.value = value.slice(0, 3) + ' ' + value.slice(3, 6) + ' ' + value.slice(6, 10);
    }
  }
});
