// Homepage JavaScript - Programs Slider & Quick Application Form

// Get language from URL
const urlParams = new URLSearchParams(window.location.search);
const lang = urlParams.get('lang') || 'en';

// Load programs for slider and dropdown
async function loadPrograms() {
  try {
    const response = await fetch('/api/programs');
    const data = await response.json();
    
    if (data.programs && data.programs.length > 0) {
      // Render slider with first 12 programs
      renderProgramsSlider(data.programs.slice(0, 12));
      
      // Populate program dropdown
      populateProgramDropdown(data.programs);
    }
  } catch (error) {
    console.error('Error loading programs:', error);
  }
}

// Render programs slider
function renderProgramsSlider(programs) {
  const sliderContainer = document.getElementById('programsSlider');
  
  const sliderHTML = programs.map(program => `
    <div class="slider-item">
      <div class="program-slider-card">
        <div class="program-slider-header">
          <span class="program-slider-category">${program.category}</span>
        </div>
        <h3 class="program-slider-name">${program.name_en}</h3>
        <div class="program-slider-details">
          <div class="program-slider-detail">
            <i class="fas fa-language"></i>
            <span>${program.language}</span>
          </div>
          <div class="program-slider-detail">
            <i class="fas fa-clock"></i>
            <span>${program.study_period}</span>
          </div>
          <div class="program-slider-detail">
            <i class="fas fa-graduation-cap"></i>
            <span>${program.thesis_type === 'thesis' ? 'With Thesis' : 'Non-Thesis'}</span>
          </div>
        </div>
        <div class="program-slider-fee">
          $${program.tuition_fee.toLocaleString()} <small>per year</small>
        </div>
        <a href="/apply?lang=${lang}&program=${encodeURIComponent(program.name_en)}" class="btn btn-primary btn-sm">
          Apply Now
        </a>
      </div>
    </div>
  `).join('');
  
  sliderContainer.innerHTML = sliderHTML;
  
  // Initialize Slick Carousel
  $('#programsSlider').slick({
    dots: true,
    infinite: true,
    speed: 500,
    slidesToShow: 3,
    slidesToScroll: 1,
    autoplay: true,
    autoplaySpeed: 3000,
    arrows: true,
    responsive: [
      {
        breakpoint: 1024,
        settings: {
          slidesToShow: 2,
          slidesToScroll: 1
        }
      },
      {
        breakpoint: 768,
        settings: {
          slidesToShow: 1,
          slidesToScroll: 1
        }
      }
    ]
  });
}

// Populate program dropdown
function populateProgramDropdown(programs) {
  const dropdown = document.getElementById('quickProgram');
  
  if (!dropdown) return;
  
  // Group programs by category
  const programsByCategory = {};
  programs.forEach(program => {
    if (!programsByCategory[program.category]) {
      programsByCategory[program.category] = [];
    }
    programsByCategory[program.category].push(program);
  });
  
  // Create optgroups
  let optionsHTML = '<option value="">Select a program...</option>';
  
  Object.keys(programsByCategory).sort().forEach(category => {
    optionsHTML += `<optgroup label="${category}">`;
    programsByCategory[category].forEach(program => {
      optionsHTML += `<option value="${program.name_en}">${program.name_en} ($${program.tuition_fee.toLocaleString()})</option>`;
    });
    optionsHTML += '</optgroup>';
  });
  
  dropdown.innerHTML = optionsHTML;
}

// Handle quick application form submission
document.getElementById('quickApplicationForm')?.addEventListener('submit', async (e) => {
  e.preventDefault();
  
  const form = e.target;
  const submitButton = form.querySelector('button[type="submit"]');
  const originalHTML = submitButton.innerHTML;
  
  // Disable submit button
  submitButton.disabled = true;
  submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
  
  // Get form data
  const formData = new FormData(form);
  
  // Get file names
  const passportFile = formData.get('passport');
  const diplomaFile = formData.get('diploma');
  
  // Prepare data for email
  const applicationData = {
    fullName: formData.get('fullName'),
    phone: formData.get('phone'),
    email: formData.get('email'),
    program: formData.get('program'),
    passportFileName: passportFile ? passportFile.name : 'No file',
    diplomaFileName: diplomaFile ? diplomaFile.name : 'No file'
  };
  
  try {
    // Send to API endpoint
    const response = await fetch('/api/quick-application', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(applicationData)
    });
    
    const data = await response.json();
    
    if (response.ok && data.success) {
      // Show success message
      alert('✅ Application submitted successfully!\n\nYour application has been sent to yazardefteri@gmail.com\n\nWe will contact you soon via phone or email.');
      
      // Reset form
      form.reset();
      
      // Scroll to top
      window.scrollTo({ top: 0, behavior: 'smooth' });
    } else {
      throw new Error(data.error || 'Unknown error');
    }
  } catch (error) {
    console.error('Error submitting application:', error);
    alert('❌ Error submitting application.\n\nPlease try again or contact us directly:\n\nWhatsApp: 0543 344 58 87\nPhone: 0212 244 66 00');
  } finally {
    submitButton.disabled = false;
    submitButton.innerHTML = originalHTML;
  }
});

// Load programs on page load
document.addEventListener('DOMContentLoaded', () => {
  loadPrograms();
});
