// Programs page JavaScript

// Get language from URL
const urlParams = new URLSearchParams(window.location.search);
const lang = urlParams.get('lang') || 'en';

// Translation keys for programs page
const translations = {
  en: {
    all: 'All Programs',
    engineering: 'Engineering',
    business: 'Business',
    law: 'Law',
    architecture: 'Architecture & Design',
    education: 'Education',
    psychology: 'Psychology',
    media: 'Media & Communication',
    social: 'Social Sciences',
    political: 'Political Science',
    arts: 'Arts',
    tuition: 'Tuition Fee',
    language: 'Language',
    duration: 'Duration',
    attendance: 'Attendance',
    thesis: 'With Thesis',
    nonThesis: 'Non-Thesis',
    perYear: 'per year',
    loading: 'Loading programs...',
    error: 'Error loading programs'
  }
};

const tr = translations[lang] || translations.en;

// State
let allPrograms = [];
let filteredPrograms = [];
let currentCategory = 'all';

// Initialize
document.addEventListener('DOMContentLoaded', () => {
  loadPrograms();
});

// Load programs from API
async function loadPrograms() {
  try {
    const response = await fetch('/api/programs');
    const data = await response.json();
    
    if (data.programs) {
      allPrograms = data.programs;
      filteredPrograms = allPrograms;
      renderFilters();
      renderPrograms();
    }
  } catch (error) {
    console.error('Error loading programs:', error);
    document.getElementById('programs-container').innerHTML = `
      <div class="error">${tr.error}</div>
    `;
  }
}

// Render filter buttons
function renderFilters() {
  // Get unique categories
  const categories = ['all', ...new Set(allPrograms.map(p => p.category))];
  
  const filtersHTML = `
    <div class="programs-filters">
      ${categories.map(cat => `
        <button 
          class="filter-btn ${cat === currentCategory ? 'active' : ''}"
          onclick="filterByCategory('${cat}')"
        >
          ${getCategoryName(cat)}
        </button>
      `).join('')}
    </div>
  `;
  
  const programsContainer = document.getElementById('programs-container');
  programsContainer.innerHTML = filtersHTML + '<div class="programs-grid" id="programsGrid"></div>';
}

// Get category name in current language
function getCategoryName(category) {
  const categoryMap = {
    'all': tr.all,
    'Engineering': tr.engineering,
    'Business': tr.business,
    'Law': tr.law,
    'Architecture & Design': tr.architecture,
    'Education': tr.education,
    'Psychology': tr.psychology,
    'Media & Communication': tr.media,
    'Social Sciences': tr.social,
    'Political Science': tr.political,
    'Arts': tr.arts
  };
  
  return categoryMap[category] || category;
}

// Filter programs by category
function filterByCategory(category) {
  currentCategory = category;
  
  if (category === 'all') {
    filteredPrograms = allPrograms;
  } else {
    filteredPrograms = allPrograms.filter(p => p.category === category);
  }
  
  renderFilters();
  renderPrograms();
}

// Render programs
function renderPrograms() {
  const programsGrid = document.getElementById('programsGrid');
  
  if (filteredPrograms.length === 0) {
    programsGrid.innerHTML = '<div class="loading">No programs found</div>';
    return;
  }
  
  programsGrid.innerHTML = filteredPrograms.map(program => `
    <div class="program-card">
      <div class="program-header">
        <h3 class="program-name">${program.name_en}</h3>
        <span class="program-category">${program.category}</span>
      </div>
      
      <div class="program-details">
        <div class="program-detail">
          <span><strong>${tr.language}:</strong></span>
          <span>${program.language}</span>
        </div>
        <div class="program-detail">
          <span><strong>Type:</strong></span>
          <span>${program.thesis_type === 'thesis' ? tr.thesis : tr.nonThesis}</span>
        </div>
        <div class="program-detail">
          <span><strong>${tr.duration}:</strong></span>
          <span>${program.study_period}</span>
        </div>
        <div class="program-detail">
          <span><strong>${tr.attendance}:</strong></span>
          <span>${program.attendance}</span>
        </div>
      </div>
      
      <div class="program-fee">
        $${program.tuition_fee.toLocaleString()} <small>${tr.perYear}</small>
      </div>
    </div>
  `).join('');
}
