import { Hono } from 'hono'
import { cors } from 'hono/cors'
import { serveStatic } from 'hono/cloudflare-workers'
import { renderer } from './renderer'
import { translations, type Language } from './translations'

type Bindings = {
  DB: D1Database;
  R2: R2Bucket;
}

const app = new Hono<{ Bindings: Bindings }>()

// Enable CORS for API routes
app.use('/api/*', cors())

// Serve static files
app.use('/static/*', serveStatic({ root: './public' }))

// Use renderer middleware
app.use(renderer)

// Get language from query parameter or default to English
function getLang(c: any): Language {
  const lang = c.req.query('lang') || 'en';
  const validLangs: Language[] = ['en', 'tk', 'fa', 'fr', 'az', 'kz', 'id'];
  return validLangs.includes(lang as Language) ? (lang as Language) : 'en';
}

// Helper function to get translations
function t(lang: Language) {
  return translations[lang];
}

// Homepage
app.get('/', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      {/* Hero Section */}
      <section class="hero-section">
        <div class="container">
          <div class="hero-content">
            <h1 class="hero-title">{tr.hero_title}</h1>
            <p class="hero-subtitle">{tr.hero_subtitle}</p>
            <div class="hero-cta">
              <a href={`/apply?lang=${lang}`} class="btn btn-primary btn-lg">
                {tr.hero_cta_apply}
              </a>
              <a href={`/residence?lang=${lang}`} class="btn btn-secondary btn-lg">
                {tr.hero_cta_check}
              </a>
            </div>
          </div>
        </div>
      </section>

      {/* Key Benefits */}
      <section class="benefits-section">
        <div class="container">
          <h2 class="section-title">{tr.benefits_title}</h2>
          <div class="benefits-grid">
            <div class="benefit-card">
              <div class="benefit-icon">
                <i class="fas fa-id-card"></i>
              </div>
              <h3 class="benefit-title">{tr.benefit_residence_title}</h3>
              <p class="benefit-desc">{tr.benefit_residence_desc}</p>
            </div>
            
            <div class="benefit-card">
              <div class="benefit-icon">
                <i class="fas fa-calendar-alt"></i>
              </div>
              <h3 class="benefit-title">{tr.benefit_flexible_title}</h3>
              <p class="benefit-desc">{tr.benefit_flexible_desc}</p>
            </div>
            
            <div class="benefit-card">
              <div class="benefit-icon">
                <i class="fas fa-briefcase"></i>
              </div>
              <h3 class="benefit-title">{tr.benefit_work_title}</h3>
              <p class="benefit-desc">{tr.benefit_work_desc}</p>
            </div>
            
            <div class="benefit-card">
              <div class="benefit-icon">
                <i class="fas fa-graduation-cap"></i>
              </div>
              <h3 class="benefit-title">{tr.benefit_diploma_title}</h3>
              <p class="benefit-desc">{tr.benefit_diploma_desc}</p>
            </div>
          </div>
        </div>
      </section>

      {/* Programs Slider Section */}
      <section class="programs-slider-section">
        <div class="container">
          <h2 class="section-title">{tr.programs_title}</h2>
          <p class="section-subtitle">Popular Master's Programs</p>
          
          <div id="programsSlider" class="programs-slider">
            <div class="loading">
              <i class="fas fa-spinner fa-spin"></i> {tr.loading}
            </div>
          </div>
          
          <div class="text-center" style="margin-top: 2rem;">
            <a href={`/programs?lang=${lang}`} class="btn btn-secondary">
              View All 55+ Programs
            </a>
          </div>
        </div>
      </section>

      {/* Quick Application Form */}
      <section class="quick-apply-section">
        <div class="container">
          <h2 class="section-title">Quick Application</h2>
          <p class="section-subtitle">Apply in 2 minutes - Just upload your documents and select program</p>
          
          <div class="quick-apply-form-container">
            <form id="quickApplicationForm" class="quick-apply-form">
              <div class="form-row">
                <div class="form-group">
                  <label for="quickFullName">Full Name *</label>
                  <input type="text" id="quickFullName" name="fullName" required />
                </div>
                
                <div class="form-group">
                  <label for="quickPhone">Phone Number *</label>
                  <input type="tel" id="quickPhone" name="phone" required placeholder="+90 XXX XXX XX XX" />
                </div>
              </div>
              
              <div class="form-row">
                <div class="form-group">
                  <label for="quickEmail">Email *</label>
                  <input type="email" id="quickEmail" name="email" required />
                </div>
                
                <div class="form-group">
                  <label for="quickProgram">Select Program *</label>
                  <select id="quickProgram" name="program" required>
                    <option value="">Loading programs...</option>
                  </select>
                </div>
              </div>
              
              <div class="form-row">
                <div class="form-group">
                  <label for="quickPassport">Passport Copy *</label>
                  <input type="file" id="quickPassport" name="passport" accept=".pdf,.jpg,.jpeg,.png" required />
                  <small>Upload passport copy (PDF, JPG, PNG)</small>
                </div>
                
                <div class="form-group">
                  <label for="quickDiploma">Diploma Copy *</label>
                  <input type="file" id="quickDiploma" name="diploma" accept=".pdf,.jpg,.jpeg,.png" required />
                  <small>Upload bachelor's diploma (PDF, JPG, PNG)</small>
                </div>
              </div>
              
              <div class="form-actions">
                <button type="submit" class="btn btn-primary btn-lg">
                  <i class="fas fa-paper-plane"></i> Submit Application
                </button>
              </div>
              
              <div class="form-note">
                <i class="fas fa-info-circle"></i>
                Your application will be sent to yazardefteri@gmail.com for review
              </div>
            </form>
          </div>
        </div>
      </section>

      {/* Quick Links */}
      <section class="quick-links-section">
        <div class="container">
          <div class="quick-links-grid">
            <a href={`/why-turkey?lang=${lang}`} class="quick-link-card">
              <i class="fas fa-question-circle"></i>
              <h3>{tr.nav_why_turkey}</h3>
              <p>{tr.learn_more}</p>
            </a>
            
            <a href={`/maltepe?lang=${lang}`} class="quick-link-card">
              <i class="fas fa-university"></i>
              <h3>{tr.nav_maltepe}</h3>
              <p>{tr.learn_more}</p>
            </a>
            
            <a href={`/programs?lang=${lang}`} class="quick-link-card">
              <i class="fas fa-book"></i>
              <h3>{tr.nav_programs}</h3>
              <p>55+ {tr.nav_programs}</p>
            </a>
            
            <a href={`/living?lang=${lang}`} class="quick-link-card">
              <i class="fas fa-map-marker-alt"></i>
              <h3>{tr.nav_living}</h3>
              <p>{tr.learn_more}</p>
            </a>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section class="cta-section">
        <div class="container">
          <div class="cta-content">
            <h2>{tr.apply_title}</h2>
            <p>{tr.apply_subtitle}</p>
            <a href={`/apply?lang=${lang}`} class="btn btn-primary btn-lg">
              {tr.apply_submit}
            </a>
          </div>
        </div>
      </section>
      
      <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
      <script src="https://cdn.jsdelivr.net/npm/slick-carousel@1.8.1/slick/slick.min.js"></script>
      <script src="/static/homepage.js"></script>
    </div>
  );
})

// Why Turkey Page
app.get('/why-turkey', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      <section class="page-header">
        <div class="container">
          <h1>{tr.why_turkey_title}</h1>
        </div>
      </section>

      <section class="content-section">
        <div class="container">
          <div class="content-grid">
            <div class="content-card">
              <h2>{tr.why_legal_title}</h2>
              <p>{tr.why_legal_desc}</p>
            </div>
            
            <div class="content-card">
              <h2>{tr.why_difference_title}</h2>
              <p>{tr.why_difference_desc}</p>
            </div>
            
            <div class="content-card">
              <h2>{tr.why_sustainable_title}</h2>
              <p>{tr.why_sustainable_desc}</p>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
})

// About Maltepe Page
app.get('/maltepe', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      <section class="page-header">
        <div class="container">
          <h1>{tr.maltepe_title}</h1>
          <p class="lead">{tr.maltepe_desc}</p>
        </div>
      </section>

      <section class="content-section">
        <div class="container">
          <div class="features-grid">
            <div class="feature-card">
              <div class="feature-icon">
                <i class="fas fa-building"></i>
              </div>
              <h3>{tr.maltepe_campus_title}</h3>
              <p>{tr.maltepe_campus_desc}</p>
            </div>
            
            <div class="feature-card">
              <div class="feature-icon">
                <i class="fas fa-globe"></i>
              </div>
              <h3>{tr.maltepe_international_title}</h3>
              <p>{tr.maltepe_international_desc}</p>
            </div>
            
            <div class="feature-card">
              <div class="feature-icon">
                <i class="fas fa-certificate"></i>
              </div>
              <h3>{tr.maltepe_accreditation_title}</h3>
              <p>{tr.maltepe_accreditation_desc}</p>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
})

// Programs Page
app.get('/programs', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      <section class="page-header">
        <div class="container">
          <h1>{tr.programs_title}</h1>
          <p class="lead">{tr.programs_subtitle}</p>
        </div>
      </section>

      <section class="content-section">
        <div class="container">
          <div id="programs-container">
            <div class="loading">
              <i class="fas fa-spinner fa-spin"></i> {tr.loading}
            </div>
          </div>
        </div>
      </section>
      
      <script src="/static/programs.js"></script>
    </div>
  );
})

// Living in Istanbul Page
app.get('/living', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      <section class="page-header">
        <div class="container">
          <h1>{tr.living_title}</h1>
          <p class="lead">{tr.living_subtitle}</p>
        </div>
      </section>

      <section class="content-section">
        <div class="container">
          <div class="cost-card">
            <h2>{tr.living_costs_title}</h2>
            <div class="cost-items">
              <div class="cost-item">
                <span class="cost-label">{tr.living_accommodation}</span>
                <span class="cost-value">$200-400</span>
              </div>
              <div class="cost-item">
                <span class="cost-label">{tr.living_food}</span>
                <span class="cost-value">$150-250</span>
              </div>
              <div class="cost-item">
                <span class="cost-label">{tr.living_transport}</span>
                <span class="cost-value">$30-50</span>
              </div>
              <div class="cost-item">
                <span class="cost-label">{tr.living_personal}</span>
                <span class="cost-value">$100-200</span>
              </div>
              <div class="cost-item cost-total">
                <span class="cost-label">{tr.living_total}</span>
                <span class="cost-value">$480-900</span>
              </div>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
})

// Residence Process Page
app.get('/residence', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      <section class="page-header">
        <div class="container">
          <h1>{tr.residence_title}</h1>
          <p class="lead">{tr.residence_subtitle}</p>
        </div>
      </section>

      <section class="content-section">
        <div class="container">
          <div class="steps-container">
            <div class="step-card">
              <div class="step-number">1</div>
              <h3>{tr.residence_step1_title}</h3>
              <p>{tr.residence_step1_desc}</p>
            </div>
            
            <div class="step-card">
              <div class="step-number">2</div>
              <h3>{tr.residence_step2_title}</h3>
              <p>{tr.residence_step2_desc}</p>
            </div>
            
            <div class="step-card">
              <div class="step-number">3</div>
              <h3>{tr.residence_step3_title}</h3>
              <p>{tr.residence_step3_desc}</p>
            </div>
            
            <div class="step-card">
              <div class="step-number">4</div>
              <h3>{tr.residence_step4_title}</h3>
              <p>{tr.residence_step4_desc}</p>
            </div>
            
            <div class="step-card">
              <div class="step-number">5</div>
              <h3>{tr.residence_step5_title}</h3>
              <p>{tr.residence_step5_desc}</p>
            </div>
            
            <div class="step-card">
              <div class="step-number">6</div>
              <h3>{tr.residence_step6_title}</h3>
              <p>{tr.residence_step6_desc}</p>
            </div>
          </div>
          
          <div class="disclaimer-box">
            <i class="fas fa-info-circle"></i>
            <p>{tr.residence_legal_note}</p>
          </div>
        </div>
      </section>
    </div>
  );
})

// Contact Page
app.get('/contact', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      <section class="page-header">
        <div class="container">
          <h1>{tr.contact_title}</h1>
          <p class="lead">{tr.contact_subtitle}</p>
        </div>
      </section>

      <section class="content-section">
        <div class="container">
          <div class="contact-grid">
            <div class="contact-info-card">
              <h2>{tr.contact_office_title}</h2>
              
              <div class="contact-info-item">
                <i class="fas fa-map-marker-alt"></i>
                <div>
                  <strong>{tr.contact_office_address}</strong>
                  <p>İstiklal Street No:49/5<br/>Taksim, İstanbul, Turkey</p>
                </div>
              </div>
              
              <div class="contact-info-item">
                <i class="fas fa-phone"></i>
                <div>
                  <strong>{tr.contact_office_phone}</strong>
                  <p><a href="tel:+902122446600">0212 244 66 00</a></p>
                </div>
              </div>
              
              <div class="contact-info-item">
                <i class="fab fa-whatsapp"></i>
                <div>
                  <strong>{tr.contact_office_whatsapp}</strong>
                  <p><a href="https://wa.me/905433445887" target="_blank">0543 344 58 87</a></p>
                </div>
              </div>
              
              <div class="contact-info-item">
                <i class="fas fa-clock"></i>
                <div>
                  <strong>{tr.contact_office_hours}</strong>
                  <p>
                    {tr.contact_hours_weekday}<br/>
                    {tr.contact_hours_saturday}<br/>
                    {tr.contact_hours_sunday}
                  </p>
                </div>
              </div>
            </div>
            
            <div class="contact-map-card">
              <h2>{tr.contact_map_title}</h2>
              <div class="map-container">
                <iframe
                  src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3009.6917890785744!2d28.98636!3d41.0362!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x0%3A0x0!2zNDHCsDAyJzEwLjMiTiAyOMKwNTknMTAuOSJF!5e0!3m2!1sen!2str!4v1234567890"
                  width="100%"
                  height="400"
                  style="border:0; border-radius: 0.75rem;"
                  allowfullscreen=""
                  loading="lazy"
                  referrerpolicy="no-referrer-when-downgrade"
                ></iframe>
              </div>
              <div class="map-cta">
                <a href="https://wa.me/905433445887" class="btn btn-primary" target="_blank">
                  <i class="fab fa-whatsapp"></i> WhatsApp: 0543 344 58 87
                </a>
                <a href="tel:+902122446600" class="btn btn-secondary">
                  <i class="fas fa-phone"></i> 0212 244 66 00
                </a>
              </div>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
})

// Application Page
app.get('/apply', (c) => {
  const lang = getLang(c);
  const tr = t(lang);
  
  return c.render(
    <div>
      <section class="page-header">
        <div class="container">
          <h1>{tr.apply_title}</h1>
          <p class="lead">{tr.apply_subtitle}</p>
        </div>
      </section>

      <section class="content-section">
        <div class="container">
          <div class="application-form-container">
            <form id="applicationForm" class="application-form">
              <div class="form-section">
                <h3>{tr.apply_personal_info}</h3>
                
                <div class="form-group">
                  <label for="fullName">{tr.apply_full_name} *</label>
                  <input type="text" id="fullName" name="fullName" required />
                </div>
                
                <div class="form-row">
                  <div class="form-group">
                    <label for="email">{tr.apply_email} *</label>
                    <input type="email" id="email" name="email" required />
                  </div>
                  
                  <div class="form-group">
                    <label for="phone">{tr.apply_phone}</label>
                    <input type="tel" id="phone" name="phone" />
                  </div>
                </div>
                
                <div class="form-group">
                  <label for="nationality">{tr.apply_nationality} *</label>
                  <input type="text" id="nationality" name="nationality" required />
                </div>
              </div>
              
              <div class="form-section">
                <h3>{tr.apply_residence_info}</h3>
                
                <div class="form-group">
                  <label for="residenceStatus">{tr.apply_residence_status} *</label>
                  <select id="residenceStatus" name="residenceStatus" required>
                    <option value="">{tr.loading}</option>
                    <option value="tourist">{tr.apply_residence_tourist}</option>
                    <option value="work">{tr.apply_residence_work}</option>
                    <option value="family">{tr.apply_residence_family}</option>
                    <option value="student">{tr.apply_residence_student}</option>
                    <option value="other">{tr.apply_residence_other}</option>
                  </select>
                </div>
                
                <div class="form-group">
                  <label for="expiryDate">{tr.apply_expiry_date}</label>
                  <input type="date" id="expiryDate" name="expiryDate" />
                </div>
              </div>
              
              <div class="form-section">
                <h3>{tr.apply_education_info}</h3>
                
                <div class="form-group">
                  <label for="educationLevel">{tr.apply_education_level} *</label>
                  <select id="educationLevel" name="educationLevel" required>
                    <option value="">{tr.loading}</option>
                    <option value="bachelor">{tr.apply_bachelor}</option>
                  </select>
                </div>
                
                <div class="form-row">
                  <div class="form-group">
                    <label for="bachelorUniversity">{tr.apply_bachelor_university}</label>
                    <input type="text" id="bachelorUniversity" name="bachelorUniversity" />
                  </div>
                  
                  <div class="form-group">
                    <label for="bachelorField">{tr.apply_bachelor_field}</label>
                    <input type="text" id="bachelorField" name="bachelorField" />
                  </div>
                </div>
                
                <div class="form-group">
                  <label for="graduationYear">{tr.apply_graduation_year}</label>
                  <input type="number" id="graduationYear" name="graduationYear" min="1980" max="2026" />
                </div>
              </div>
              
              <div class="form-section">
                <h3>{tr.apply_program_preference}</h3>
                
                <div class="form-group">
                  <label for="programInterest">{tr.apply_program_interest} *</label>
                  <select id="programInterest" name="programInterest" required>
                    <option value="">{tr.loading}</option>
                    <option value="engineering">{tr.programs_filter_engineering}</option>
                    <option value="business">{tr.programs_filter_business}</option>
                    <option value="law">{tr.programs_filter_law}</option>
                    <option value="architecture">{tr.programs_filter_architecture}</option>
                    <option value="education">{tr.programs_filter_education}</option>
                    <option value="psychology">{tr.programs_filter_psychology}</option>
                    <option value="media">{tr.programs_filter_media}</option>
                    <option value="social">{tr.programs_filter_social}</option>
                    <option value="political">{tr.programs_filter_political}</option>
                    <option value="arts">{tr.programs_filter_arts}</option>
                  </select>
                </div>
                
                <div class="form-group">
                  <label for="languagePreference">{tr.apply_language_preference} *</label>
                  <select id="languagePreference" name="languagePreference" required>
                    <option value="">{tr.loading}</option>
                    <option value="turkish">{tr.apply_turkish}</option>
                    <option value="english">{tr.apply_english}</option>
                  </select>
                </div>
                
                <div class="form-group">
                  <label for="message">{tr.apply_message}</label>
                  <textarea id="message" name="message" rows="4"></textarea>
                </div>
              </div>
              
              <div class="form-actions">
                <button type="submit" class="btn btn-primary btn-lg">
                  {tr.apply_submit}
                </button>
              </div>
              
              <div class="form-privacy">
                <i class="fas fa-lock"></i> {tr.apply_privacy}
              </div>
            </form>
          </div>
        </div>
      </section>
      
      <script src="/static/application.js"></script>
    </div>
  );
})

// API: Get all programs
app.get('/api/programs', async (c) => {
  try {
    const { env } = c;
    const category = c.req.query('category') || '';
    const language = c.req.query('language') || '';
    const thesisType = c.req.query('thesis_type') || '';
    
    let query = 'SELECT * FROM programs WHERE 1=1';
    const params: any[] = [];
    
    if (category) {
      query += ' AND category = ?';
      params.push(category);
    }
    
    if (language) {
      query += ' AND language = ?';
      params.push(language);
    }
    
    if (thesisType) {
      query += ' AND thesis_type = ?';
      params.push(thesisType);
    }
    
    query += ' ORDER BY category, name_en';
    
    const { results } = await env.DB.prepare(query).bind(...params).all();
    
    return c.json({ programs: results });
  } catch (error: any) {
    return c.json({ error: error.message }, 500);
  }
});

// API: Submit application
// Helper function to send email notification
async function sendEmailNotification(data: any, type: 'full' | 'quick') {
  const emailContent = type === 'quick' ? 
    `
    🎓 YENI HIZLI BAŞVURU / NEW QUICK APPLICATION
    
    📝 Başvuru Bilgileri / Application Information:
    ----------------------------------------------------
    👤 Ad Soyad / Full Name: ${data.fullName}
    📧 Email: ${data.email}
    📱 Telefon / Phone: ${data.phone}
    🎯 Seçilen Program / Selected Program: ${data.program}
    
    📎 Belgeler / Documents:
    - Pasaport / Passport: ${data.passportFileName}
    - Diploma / Diploma: ${data.diplomaFileName}
    
    ⏰ Başvuru Tarihi / Application Date: ${new Date().toLocaleString('tr-TR')}
    
    ℹ️ Bu hızlı başvurudur. Belgeler yüklenmemiştir, sadece bilgi formudur.
    This is a quick application. Documents are not uploaded, just information form.
    
    Lütfen başvuru sahibi ile iletişime geçin.
    Please contact the applicant.
    `
    :
    `
    🎓 YENI TAM BAŞVURU / NEW FULL APPLICATION
    
    📝 Başvuru Bilgileri / Application Information:
    ----------------------------------------------------
    👤 Ad Soyad / Full Name: ${data.fullName}
    📧 Email: ${data.email}
    📱 Telefon / Phone: ${data.phone || 'Not provided'}
    🌍 Uyruk / Nationality: ${data.nationality}
    
    🏠 İkamet Bilgileri / Residence Information:
    - Mevcut Durum / Current Status: ${data.residenceStatus}
    - Bitiş Tarihi / Expiry Date: ${data.expiryDate || 'Not provided'}
    
    🎓 Eğitim Bilgileri / Education Information:
    - Seviye / Level: ${data.educationLevel}
    - Üniversite / University: ${data.bachelorUniversity || 'Not provided'}
    - Bölüm / Field: ${data.bachelorField || 'Not provided'}
    - Mezuniyet Yılı / Graduation Year: ${data.graduationYear || 'Not provided'}
    
    🎯 Program Tercihleri / Program Preferences:
    - İlgi Alanı / Interest: ${data.programInterest}
    - Dil / Language: ${data.languagePreference}
    
    💬 Mesaj / Message:
    ${data.message || 'No message'}
    
    ⏰ Başvuru Tarihi / Application Date: ${new Date().toLocaleString('tr-TR')}
    `;

  // For Cloudflare Workers, we would typically use a service like SendGrid, Mailgun, or Resend
  // For now, we'll just log and save to database
  // In production, you would add email service integration here
  console.log('Email notification:', emailContent);
  
  // Return success - in production you would return the email service response
  return { success: true };
}

app.post('/api/applications', async (c) => {
  try {
    const { env } = c;
    const data = await c.req.json();
    
    const result = await env.DB.prepare(`
      INSERT INTO applications (
        full_name, email, phone, nationality, current_residence_status,
        residence_expiry_date, education_level, bachelor_university,
        bachelor_field, bachelor_graduation_year, program_interest,
        language_preference, message
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `).bind(
      data.fullName,
      data.email,
      data.phone || null,
      data.nationality,
      data.residenceStatus,
      data.expiryDate || null,
      data.educationLevel,
      data.bachelorUniversity || null,
      data.bachelorField || null,
      data.graduationYear || null,
      data.programInterest,
      data.languagePreference,
      data.message || null
    ).run();
    
    // Send email notification
    await sendEmailNotification(data, 'full');
    
    return c.json({
      success: true,
      applicationId: result.meta.last_row_id,
      message: 'Application submitted successfully. Email notification sent to yazardefteri@gmail.com'
    });
  } catch (error: any) {
    return c.json({ error: error.message }, 500);
  }
});

// API: Quick Application (Homepage form)
app.post('/api/quick-application', async (c) => {
  try {
    const { env } = c;
    const data = await c.req.json();
    
    // Save to applications table with all required fields
    const result = await env.DB.prepare(`
      INSERT INTO applications (
        full_name, email, phone, nationality, current_residence_status, 
        education_level, program_interest, language_preference, message
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `).bind(
      data.fullName,
      data.email,
      data.phone,
      'To be determined', // Placeholder for nationality
      'Quick Application', // Placeholder for residence status
      'Bachelor', // Placeholder for education level
      data.program,
      'English', // Placeholder for language preference
      `Quick Application - Files: Passport: ${data.passportFileName}, Diploma: ${data.diplomaFileName}`
    ).run();
    
    // Send email notification
    await sendEmailNotification(data, 'quick');
    
    return c.json({
      success: true,
      applicationId: result.meta.last_row_id,
      message: 'Quick application submitted successfully. Email notification sent to yazardefteri@gmail.com'
    });
  } catch (error: any) {
    return c.json({ error: error.message }, 500);
  }
});

export default app
